// Copyright 2019 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Package automl contains samples for Google Cloud AutoML API v1.
package automl

// [START automl_list_models]
import (
	"context"
	"fmt"
	"io"

	automl "cloud.google.com/go/automl/apiv1"
	"cloud.google.com/go/automl/apiv1/automlpb"
	"google.golang.org/api/iterator"
)

// listModels lists existing models.
func listModels(w io.Writer, projectID string, location string) error {
	// projectID := "my-project-id"
	// location := "us-central1"

	ctx := context.Background()
	client, err := automl.NewClient(ctx)
	if err != nil {
		return fmt.Errorf("NewClient: %w", err)
	}
	defer client.Close()

	req := &automlpb.ListModelsRequest{
		Parent: fmt.Sprintf("projects/%s/locations/%s", projectID, location),
	}

	it := client.ListModels(ctx, req)

	// Iterate over all results
	for {
		model, err := it.Next()
		if err == iterator.Done {
			break
		}
		if err != nil {
			return fmt.Errorf("ListModels.Next: %w", err)
		}

		// Retrieve deployment state.
		deploymentState := "undeployed"
		if model.GetDeploymentState() == automlpb.Model_DEPLOYED {
			deploymentState = "deployed"
		}

		// Display the model information.
		fmt.Fprintf(w, "Model name: %v\n", model.GetName())
		fmt.Fprintf(w, "Model display name: %v\n", model.GetDisplayName())
		fmt.Fprintf(w, "Model create time:\n")
		fmt.Fprintf(w, "\tseconds: %v\n", model.GetCreateTime().GetSeconds())
		fmt.Fprintf(w, "\tnanos: %v\n", model.GetCreateTime().GetNanos())
		fmt.Fprintf(w, "Model deployment state: %v\n", deploymentState)
	}

	return nil
}

// [END automl_list_models]
